#include "downloads.h"
#include <QUrl>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QFile>
#include <QDir>
#include <QDebug>
#include <QFileInfo>
#include <QMessageBox>
#include <QApplication>

Downloads::Downloads(QUrl url, QObject *parent) : m_url(url),
    m_progress(0),
    m_total(0),
    m_wasAborted(false),
    QNetworkAccessManager(parent)
{
    m_file = new QFile(m_url.fileName(),this);
    m_file->open(QFile::WriteOnly);
    startRequest(m_url);
}

void Downloads::startRequest(const QUrl &requestedUrl)
{
    m_url = requestedUrl;

    m_reply = get(QNetworkRequest(m_url));
    connect(m_reply, &QNetworkReply::finished, this, &Downloads::replyFinished);
    connect(m_reply, &QIODevice::readyRead, this, &Downloads::httpReadyRead);
    connect(m_reply, SIGNAL(downloadProgress(qint64,qint64)), this,SLOT(handleProgress(qint64,qint64)));

}

void Downloads::replyFinished()
{
    QFileInfo fi;
    if (m_file) {
        fi.setFile(m_file->fileName());
        m_file->close();
        delete m_file;
        m_file = Q_NULLPTR;
    }

    if (m_reply->error()) {
        QFile::remove(fi.absoluteFilePath());
        m_reply->deleteLater();
        m_reply = Q_NULLPTR;
        return;
    }
    const QVariant redirectionTarget = m_reply->attribute(QNetworkRequest::RedirectionTargetAttribute);

    m_reply->deleteLater();
    m_reply = Q_NULLPTR;

    if (!redirectionTarget.isNull()) {
        const QUrl redirectedUrl = m_url.resolved(redirectionTarget.toUrl());
        m_file = new QFile(redirectedUrl.fileName(),this);
        m_file->open(QFile::WriteOnly);
        startRequest(redirectedUrl);
    }

}

void Downloads::abortDownload()
{
    m_reply->abort();
    setWasAborted(true);
}

void Downloads::handleProgress(qint64 current, qint64 total)
{
    m_progress = current;
    m_total = total;
    emit downloadProgress(m_progress, m_total);
}

qint64 Downloads::total() const
{
    return m_total;
}

void Downloads::setTotal(const qint64 &total)
{
    m_total = total;
}

qint64 Downloads::progress() const
{
    return m_progress;
}

void Downloads::setProgress(const qint64 &progress)
{
    m_progress = progress;
}

void Downloads::setUrl(const QUrl &url)
{
    m_url = url;
}

QUrl Downloads::url() const
{
    return m_url;
}


void Downloads::httpReadyRead()
{

    if (m_file)
        m_file->write(m_reply->readAll());
}

bool Downloads::wasAborted() const
{
    return m_wasAborted;
}

void Downloads::setWasAborted(bool wasAborted)
{
    m_wasAborted = wasAborted;
}