#include "inventorycborreader.h"
#include <QByteArray>
#include <QFile>

InventoryCborReader::InventoryCborReader()
{

}

// Function to assign values in the current inventory item.
// In this example application, strictly speaking, keeping
// track of the current key isn't necessary because each type
// is associated with only one attribute of the inventory item.
// For a more complex structure though you may need to distinguish
// between values for different attributes.
void InventoryCborReader::assignStringValue(QString value)
{
    if (m_currentKey == "Name")
        m_currentItem.setName(value);

    m_currentKey = QString();
}

void InventoryCborReader::assignDoubleValue(double value)
{
    if (m_currentKey == "Price")
        m_currentItem.setPrice(value);

    m_currentKey = QString();
}

void InventoryCborReader::assignUnsignedIntegerValue(quint64 value)
{
    if (m_currentKey == "Quantity")
        m_currentItem.setQuantity(value);

    m_currentKey = QString();
}

void InventoryCborReader::assignNegativeIntegerValue(qint64 value)
{
    if (m_currentKey == "Quantity")
        m_currentItem.setQuantity(value);

    m_currentKey = QString();
}

void InventoryCborReader::assignBooleanValue(bool value)
{
    if (m_currentKey == "Used")
        m_currentItem.setUsed(value);

    m_currentKey = QString();
}

void InventoryCborReader::assignUuid(QUuid value)
{
    if (m_currentKey == "ID")
        m_currentItem.setID(value);

    m_currentKey = QString();
}

bool InventoryCborReader::readCborData(QCborStreamReader* reader)
{
    while (!reader->lastError() && reader->hasNext())
    {
        QCborStreamReader::Type type = reader->type();

        // Handle some basic types
        if (type == QCborStreamReader::UnsignedInteger)
        {
            quint64 num = reader->toUnsignedInteger();
            assignUnsignedIntegerValue(num);
            reader->next();
        }
        else if (type == QCborStreamReader::NegativeInteger)
        {
            qint64 num = reader->toInteger();
            assignNegativeIntegerValue(num);
            reader->next();
        }
        else if (type == QCborStreamReader::Double)
        {
            double num = reader->toDouble();
            assignDoubleValue(num);
            reader->next();
        }
        else if (type == QCborStreamReader::String)
        {
            // Read a CBOR string, concatenating all
            // the chunks into a single string.
            QString str;
            auto r = reader->readString();
            while (r.status == QCborStreamReader::Ok)
            {
                str += r.data;
                r = reader->readString();
            }

            if (r.status == QCborStreamReader::Error)
            {
                // handle error condition
                str.clear();
            }

            // If the current key is empty, this string value is a new tag
            // Otherwise, it's a value
            if (m_currentKey.isEmpty())
            {
                m_currentKey = str;
            }
            else
            {
                assignStringValue(str);
            }
        }
        else if (type == QCborStreamReader::ByteArray)
        {
            // Read a byte array. In this example application
            // we make the simplifying assumption that the
            // byte array is a UUID as that's the only data
            // represented that way.
            QByteArray uuidArray;
            auto r = reader->readByteArray();
            while (r.status == QCborStreamReader::Ok)
            {
                uuidArray.append(r.data);
                r = reader->readByteArray();
            }
            assignUuid(QUuid(uuidArray));
        }
        else if (type == QCborStreamReader::Tag)
        {
            // Get the tag type. Strictly speaking, we don't
            // need to set the current key as "ID" here
            // as there should already have been a string
            // element beforehand to set it.
            QCborTag tag = reader->toTag();
            if (tag == QCborKnownTags::Uuid)
                m_currentKey = QString("ID");
            reader->next();
        }
        else if (type == QCborStreamReader::SimpleType)
        {
            // Handle simple types; in this example, we
            // don't do anything with Nulls and Undefined
            // CBOR values
            switch (reader->toSimpleType()) {
            case QCborSimpleType::False:
                assignBooleanValue(false);
                break;
            case QCborSimpleType::True:
                assignBooleanValue(true);
                break;
            case QCborSimpleType::Null:
            case QCborSimpleType::Undefined:
                break;
            }
            reader->next();
        }
        else if (type == QCborStreamReader::Array)
        {
            // Start of item list
            reader->enterContainer();

            // Read elements until end of array is reached
            bool ret = readCborData(reader);
            if (ret)
                reader->leaveContainer();
            else return false;
        }
        else if (type == QCborStreamReader::Map)
        {
            // Start of new inventory item
            m_currentItem = InventoryItem();
            reader->enterContainer();

            // Read elements until end of map is reached
            bool ret = readCborData(reader);
            if (ret)
                reader->leaveContainer();
            else return false;

            // Successfully created new item -- add to list
            m_itemList.push_back(m_currentItem);
        }
        else
        {
            // Ignore all other types, go to the next element
            reader->next();
        }
    }

    // Return true if there were no errors
    return !reader->lastError();
}

bool InventoryCborReader::readCborFile(QString filePath)
{
    m_itemList.clear();
    m_currentKey = QString();

    QFile file(filePath);
    bool ret = file.open(QIODevice::ReadOnly);
    if (!ret)
        return false;

    QCborStreamReader reader(&file);
    ret = readCborData(&reader);

    file.close();

    return ret;
}
