#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QAction>
#include <QFileDialog>
#include <QMessageBox>

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    // Set up inventory list model
    m_model = new InventoryListModel();
    ui->itemListView->setModel(m_model);

    // Set up UI connections
    connect(ui->addButton, &QAbstractButton::clicked, this, &MainWindow::onAddClicked);
    connect(ui->deleteButton, &QAbstractButton::clicked, this, &MainWindow::onDeleteClicked);
    connect(ui->applyButton, &QAbstractButton::clicked, this, &MainWindow::onApplyClicked);
    connect(ui->actionImport, &QAction::triggered, this, &MainWindow::onImport);
    connect(ui->actionExport, &QAction::triggered, this, &MainWindow::onExport);

    connect(ui->nameEdit, &QLineEdit::textEdited, this, &MainWindow::onItemEdited);
    connect(ui->priceEdit, &QLineEdit::textEdited, this, &MainWindow::onItemEdited);
    connect(ui->quantityEdit, &QLineEdit::textEdited, this, &MainWindow::onItemEdited);
    connect(ui->usedCheckBox, &QAbstractButton::clicked, this, &MainWindow::onCheckboxClicked);

    // Watch for when selection is changed
    connect(ui->itemListView->selectionModel(), SIGNAL(selectionChanged (const QItemSelection &, const QItemSelection &)),
            this, SLOT(selectionChangedSlot(const QItemSelection &, const QItemSelection &)));

    // Disable UI elements by default
    enableEntryUI(false);
}

MainWindow::~MainWindow()
{
    delete ui;
}

// Handle when the user clicks the Add button
// Adds a new item to the model and automatically selects it
void MainWindow::onAddClicked()
{
    InventoryItem newItem = InventoryItem("New Item", QUuid::createUuid(), 0.0, 0, false);
    m_model->addItem(newItem);
    QModelIndex newIndex = m_model->index(0, 0);
    ui->itemListView->setCurrentIndex(newIndex);
}

// Handle when the user clicks the Delete button
// Deletes the currently selected item
void MainWindow::onDeleteClicked()
{
    int row = ui->itemListView->currentIndex().row();
    m_model->deleteItem(row);
}

// Handle when the user clicks the Apply button
// Updates the current item in the model with what the
// user entered in the form
void MainWindow::onApplyClicked()
{
    int row = ui->itemListView->currentIndex().row();
    InventoryItem newItem = InventoryItem(ui->nameEdit->text(),
                                          QUuid(ui->idLabel->text()),
                                          ui->priceEdit->text().toDouble(),
                                          ui->quantityEdit->text().toInt(),
                                          ui->usedCheckBox->isChecked());
    m_model->updateItem(row, newItem);

    // Disable apply button now that changes have been applied
    ui->applyButton->setEnabled(false);
}

// Handles enabling or disabling the application's UI
// elements
void MainWindow::enableEntryUI(bool enabled)
{
    // Enable or disable form UI
    ui->nameEdit->setEnabled(enabled);
    ui->idLabel->setEnabled(enabled);
    ui->priceEdit->setEnabled(enabled);
    ui->quantityEdit->setEnabled(enabled);
    ui->usedCheckBox->setEnabled(enabled);
    ui->deleteButton->setEnabled(enabled);

    // Apply button should be disabled by default
    ui->applyButton->setEnabled(false);

    // Clear current text if disabled
    if (!enabled)
    {
        ui->nameEdit->setText(QString());
        ui->idLabel->setText(QString());
        ui->priceEdit->setText(QString());
        ui->quantityEdit->setText(QString());
        ui->usedCheckBox->setChecked(false);
    }
}

void MainWindow::onImport()
{
    QString fileName = QFileDialog::getOpenFileName(this,
                                                    tr("Open CBOR file"), "", tr("CBOR Files (*.cbor)"));

    if (!fileName.isEmpty())
    {
        bool ret = m_model->readCborFile(fileName);
        if (ret)
        {
            QModelIndex newIndex = m_model->index(0, 0);
            ui->itemListView->setCurrentIndex(newIndex);
        }
        else
        {
            QMessageBox::warning(this, tr("Error"),
                                 tr("An error occurred when reading the file %1")
                                 .arg(fileName));
        }
    }
}

void MainWindow::onExport()
{
    QString fileName = QFileDialog::getSaveFileName(this,
                                                    tr("Save CBOR file"), "", tr("CBOR Files (*.cbor)"));

    if (!fileName.isEmpty())
    {
        bool ret = m_model->writeCborFile(fileName);
        if (!ret)
            QMessageBox::warning(this, tr("Error"),
                                 tr("An error occurred when writing the file %1")
                                 .arg(fileName));
    }
}

// If the user edits an inventory item, enable the Apply button
void MainWindow::onItemEdited(const QString &text)
{
    Q_UNUSED(text);
    ui->applyButton->setEnabled(true);
}

// If the user clicks a checkbox, enable the Apply button
void MainWindow::onCheckboxClicked()
{
    ui->applyButton->setEnabled(true);
}

// Handle when the selection is changed, either by the user clicking or programmatically (like
// when an item is deleted from the model)
void MainWindow::selectionChangedSlot(const QItemSelection &selected, const QItemSelection &deselected)
{
    Q_UNUSED(selected);
    Q_UNUSED(deselected);

    int row = ui->itemListView->currentIndex().row();

    // If there is no selection (as after the last item in the model is deleted)
    // then disable the form UI
    if (row == -1)
    {
        enableEntryUI(false);
        return;
    }

    // Get the inventory item in the model and populate the UI with its data
    InventoryItem item = m_model->getItem(row);
    ui->nameEdit->setText(item.getName());
    ui->idLabel->setText(item.getID().toString());
    ui->priceEdit->setText(QString("%1").arg(item.getPrice()));
    ui->quantityEdit->setText(QString("%1").arg(item.getQuantity()));
    ui->usedCheckBox->setChecked(item.getUsed());

    enableEntryUI(true);
}
